package furny.ga.rules;

import furny.entities.Tag;
import furny.ga.FurnEntry;
import furny.ga.FurnLayoutIndividual;
import ga.core.GA;
import ga.core.validation.GAContext;
import ga.core.validation.IValidationRule;
import ga.view.interfaces.IPhenotypeSpace;

import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This is a {@link IValidationRule} implementation that keeps furnitures at a
 * wall.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class MaxDistanceToWallRule implements
    IValidationRule<FurnLayoutIndividual> {

  // the logger for this class
  private static final Logger LOGGER = Logger
      .getLogger(MaxDistanceToWallRule.class.getName());

  private float maxAllowedDistance = 1f;
  private final Tag tag;

  /**
   * Creates the rule for a given tag name and a maximum distance.
   * 
   * @param tagName
   *          The tag name.
   * @param maxDistance
   *          The maximum distance.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public MaxDistanceToWallRule(final String tagName, final float maxDistance) {
    this.tag = Tag.valueOf(tagName);
    this.maxAllowedDistance = maxDistance;
  }

  /**
   * Creates the rule with a maximum distance. The tag name will be
   * "Location/Wall".
   * 
   * @param maxDistance
   *          The maximum distance.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public MaxDistanceToWallRule(final float maxDistance) {
    this("Location/Wall", maxDistance);
  }

  @Override
  public float matchingDegree(final FurnLayoutIndividual individual,
      final GAContext context) {

    final Object o = context.get(GA.KEY_VALIDATION_SPACE);

    float maxDistance = 0f;

    if (o != null && o instanceof IPhenotypeSpace) {
      final IPhenotypeSpace space = (IPhenotypeSpace) o;

      for (final FurnEntry entry : individual.getFurnitures()) {
        if (entry.getFurniture().getMetaData().getTags().contains(tag)) {
          maxDistance = Math.max(maxDistance, space.minDistanceToWall(
              entry.getBounds(), entry.getVector().getRotation()));
        }
      }

    } else {
      throw new RuntimeException("Showroom not found in the validation context");
    }

    if (LOGGER.isLoggable(Level.FINE)) {
      LOGGER.fine("Max Distance: " + maxDistance);
    }

    // prevent division by zero
    if (maxDistance == 0f) {
      return 1f;
    }

    return Math.min(1f, maxAllowedDistance / maxDistance);
  }

  @Override
  public IValidationRule.Type getType() {
    return Type.PROPOSED;
  }

  // public static void main(final String[] args) {
  // final FurnyApp app = new FurnyApp();
  // app.setShowSettings(false);
  // app.start();
  //
  // app.enqueue(new Callable<Void>() {
  // @Override
  // public Void call() throws Exception {
  // final ShowRoom sr = new BoxShowRoom(app.getAssetManager(), app
  // .getAppSettings(), 5f, 5f, 5f);
  // sr.setLocalTranslation(0f, 0f, 0f);
  //
  // final Box b = new Box(.5f, .5f, .5f);
  // final Geometry g = new Geometry("", b);
  // g.setLocalTranslation(2.0f, 0f, 1f);
  //
  // final Polygon2D bounds = new Rectangle2D(g.getWorldTranslation().x
  // - b.getXExtent(), g.getWorldTranslation().z - b.getZExtent(), b
  // .getXExtent() * 2f, b.getZExtent() * 2f);
  //
  // sr.attachChild(g);
  //
  // // System.out.println(sr.minDistanceToWall(g));
  //
  // System.out.println(sr.minDistanceToWall(bounds, 0f));
  //
  // app.stop();
  // return null;
  // }
  // });
  // }
}
